/**
 * Copyright 2021 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Middleware function that checks to see if there is a signed-in
 * user sessions before continuing to the route. If there is not, then the user
 * is redirected to sign-in.
 */

const checkAuthenticated = (req, res, next) => {
  // Check to see if we have a user in this session.
  let isLoggedIn = !!req.session.user && !!req.session.tokens;
  if (isLoggedIn) {
    // Check the session to see if a login_hint was used. If it was, then we
    // want to check that the current signed-in user is the same as the user
    // coming from Classroom.
    const addon = req.session.addon || {};
    const login_hint = addon.login_hint;

    // If the Classroom user does not match the current add-on user, they will
    // be redirected to the standard sign-in flow.
    if (login_hint && req.session.user.sub != login_hint) {
      isLoggedIn = false;
    }
  }

  // if the currently signed user has been validated as the Classroom user
  // we may continue on with the route handling.
  if (isLoggedIn) { return next() }

  // There is either no signed in user or the wrong signed in user so we need to
  // ask them to sign in again.
  res.redirect("/signin");
};

module.exports = checkAuthenticated;
