// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.pay.echo.outbound.domain;

import java.util.UUID;

/**
 * Defines the echo request sent to Google for the "/gsp/v1/echo" api.
 *
 * @see <a href=
 *     "https://developers.google.com/standard-payments/payment-integrator-diagnostic-api/rest/v2/TopLevel/echo">
 *     Echo Request </a>
 */
public class OutboundEchoRequest {

  private OutboundRequestHeader requestHeader;
  private String clientMessage;

  public OutboundEchoRequest(Builder echoRequestBuilder) {
    this.requestHeader = echoRequestBuilder.requestHeader;
    this.clientMessage = echoRequestBuilder.clientMessage;
  }

  public static Builder builder() {
    return new Builder()
        .setRequestHeader(new OutboundRequestHeader(new OutboundProtocolVersion(), UUID.randomUUID().toString()));
  }

  public OutboundRequestHeader getRequestHeader() {
    return requestHeader;
  }

  public void setRequestHeader(OutboundRequestHeader requestHeader) {
    this.requestHeader = requestHeader;
  }

  public String getClientMessage() {
    return clientMessage;
  }

  public void setClientMessage(String clientMessage) {
    this.clientMessage = clientMessage;
  }

  public static class Builder {

    private OutboundRequestHeader requestHeader;
    private String clientMessage;

    public Builder setRequestHeader(OutboundRequestHeader requestHeader) {
      this.requestHeader = requestHeader;
      return this;
    }

    public Builder setClientMessage(String clientMessage) {
      this.clientMessage = clientMessage;
      return this;
    }

    public OutboundEchoRequest build() {
      return new OutboundEchoRequest(this);
    }
  }
}
