// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.pay.echo.logging;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.logging.Logger;
import com.google.pay.echo.outbound.service.OutboundEchoService;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.util.StreamUtils;

/** An interceptor that logs requests and responses when 'debug' log level is enabled. */
@Component
public class ClientHttpRequestInterceptorDebugger implements ClientHttpRequestInterceptor {

  private static final Logger LOGGER = Logger.getLogger(OutboundEchoService.class.getName());

  @Override
  public ClientHttpResponse intercept(
      HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
    logRequest(request, body);
    ClientHttpResponse response = execution.execute(request, body);
    logResponse(response);
    return response;
  }

  /** Logs an intercepted request. */
  private void logRequest(HttpRequest request, byte[] body) throws IOException {
      LOGGER.info(
          "===========================outbound request to Google"
              + " begin================================================");
      LOGGER.info("URI         : {}" + request.getURI());
      LOGGER.info("Method      : {}" + request.getMethod());
      LOGGER.info("Headers     : {}" + request.getHeaders());
      LOGGER.info("Request body: {}" + new String(body, StandardCharsets.UTF_8));
      LOGGER.info(
          "==========================request end================================================");
  }

  /** Logs an intercepted response. */
  private void logResponse(ClientHttpResponse response) throws IOException {
      LOGGER.info(
          "============================inbound response begin==========================================");
      LOGGER.info("Status code  : {}" + response.getStatusCode());
      LOGGER.info("Status text  : {}" + response.getStatusText());
      LOGGER.info("Headers      : {}" + response.getHeaders());
      LOGGER.info(
          "Response body: {}" +
          StreamUtils.copyToString(response.getBody(), Charset.defaultCharset()));
      LOGGER.info(
          "=======================response end=================================================");
  }
}
